package cz.frantovo.xmlWebGenerator;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Date;
import java.net.URI;
import java.net.URISyntaxException;
import static cz.frantovo.xmlWebGenerator.NástrojeCLI.*;

/**
 * Knihovna funkcí volaných z XSLT.
 *  
 * TODO:
 *	- rozdělit na více modulů (jmenných prostorů).
 *	- CLI konektor
 * 
 * @author fiki
 */
public class Funkce {

	private static final String PŘÍKAZ_PYGMENTIZE = "pygmentize";
	private static final String PŘÍKAZ_DOT = "dot";
	private static final String PŘÍKAZ_MARKDOWN = "markdown";
	private static final String ADRESÁŘ_VSTUPNÍ = "vstup";
	private static final String ADRESÁŘ_VÝSTUPNÍ = "výstup";
	private static int počítadloDiagramů = 0;

	/**
	 * Zjištuje, kdy byl naposledy daný soubor změněn.
	 * @param soubor cesta k souboru
	 * @return datum poslední změny
	 * @throws URISyntaxException
	 */
	public static Date posledníZměna(String soubor) throws URISyntaxException {
		URI uri = new URI(soubor);
		File f = new File(uri);
		return new Date(f.lastModified());
	}

	/**
	 * Zvýrazňuje syntaxi zdrojového kódu. Používá k tomu externí program/knihovnu pygmentize.
	 * @param zdroják zdrojový kód, který předáme příkazu pygmentize na standardním vstupu
	 * @param jazyk předáme příkazu pygmentize jako parametr -l &lt;lexer&gt;
	 * @return zvýrazněný text nebo null, pokud došlo k chybě.
	 * TODO: 
	 *	- vracet místo textu instanci com.icl.saxon.om.NodeInfo http://saxon.sourceforge.net/saxon6.5.3/extensibility.html
	 *  - nebo kontrolovat validitu vygenerovaného kódu (v současnosti se spoléháme na bezchybnost pygmentize)
	 */
	public static String zvýrazniSyntaxi(String zdroják, String jazyk) throws IOException, InterruptedException {
		if (jazyk == null || jazyk.length() == 0) {
			System.err.println("Není vyplněn atribut „jazyk“ → není jasné, jak se má zvýrazňovat.");
			return null;
		} else if (isPříkazDostupný(PŘÍKAZ_PYGMENTIZE)) {
			Runtime r = Runtime.getRuntime();
			Process p = r.exec(new String[]{PŘÍKAZ_PYGMENTIZE, "-f", "html", "-l", jazyk});

			PrintStream vstupProcesu = new PrintStream(p.getOutputStream());
			vstupProcesu.print(zdroják);
			vstupProcesu.close();

			String výsledek = načtiProud(p.getInputStream());
			String chyby = načtiProud(p.getErrorStream());

			p.waitFor();

			if (chyby.length() == 0) {
				// Pozor: pygmentize má i při chybě návratový kód 0 → je potřeba kontrolovat chybový výstup.
				return výsledek;
			} else {
				System.err.print("Při zvýrazňování syntaxe došlo k chybě: " + chyby);
				return null;
			}
		} else {
			System.err.println("Příkaz " + PŘÍKAZ_PYGMENTIZE + " není na vašem systému dostupný → zvýrazňování syntaxe nebude fungovat.");
			System.err.println("Můžete ho nainstalovat pomocí:");
			System.err.println("\t$ aptitude install python-pygments   # (Debian/Ubuntu)");
			System.err.println("\t$ yum install python-pygments        # (Fedora/RedHat)");
			return null;
		}
	}

	/**
	 * Vygeneruje CSS styl pro zvýrazňování syntaxe.
	 * @return obsah CSS souboru nebo null, pokud generování nebylo možné
	 */
	public static String generujCssSyntaxe() throws IOException, InterruptedException {
		if (isPříkazDostupný(PŘÍKAZ_PYGMENTIZE)) {
			Runtime r = Runtime.getRuntime();
			Process p = r.exec(new String[]{PŘÍKAZ_PYGMENTIZE, "-S", "default", "-f", "html"});
			return načtiProud(p.getInputStream());
		} else {
			return null;
		}
	}

	/**
	 * Vytvoří obrázek s diagramem.
	 * @param zadání definice diagramu ve formátu dot
	 * @param vodorovně zda má být graf orientovaný vodorovně
	 * @return název souboru bez přípony, který byl vytvořen, nebo null, pokud došlo k chybě.
	 */
	public static String vytvořDiagram(String zadání, boolean vodorovně) throws IOException, InterruptedException {
		if (isPříkazDostupný(PŘÍKAZ_DOT)) {
			počítadloDiagramů++;
			String soubor = "diagram-" + počítadloDiagramů;
			String souborSložka = ADRESÁŘ_VÝSTUPNÍ + File.separator + soubor;

			StringBuilder zdroják = new StringBuilder(zadání.length() + 200);

			zdroják.append("digraph d {\n");
			zdroják.append("\tbgcolor=\"transparent\";\n");
			if (vodorovně) {
				zdroják.append("\trankdir=LR;");
			}
			zdroják.append(zadání);
			zdroják.append("}\n");

			Runtime r = Runtime.getRuntime();
			Process p = r.exec(new String[]{PŘÍKAZ_DOT, "-T", "svg", "-o", souborSložka + ".svg"});

			/**
			 * TODO: generovat i PNG bitmapu
			 */
			PrintStream vstupProcesu = new PrintStream(p.getOutputStream());
			vstupProcesu.print(zdroják.toString());
			vstupProcesu.close();

			String chyby = načtiProud(p.getErrorStream());

			p.waitFor();

			if (chyby.length() == 0) {
				return soubor;
			} else {
				System.err.print("Při vytváření diagramu došlo k chybě: " + chyby);
				return null;
			}
		} else {
			System.err.println("Příkaz " + PŘÍKAZ_DOT + " není na vašem systému dostupný → diagramy nelze vygreslit.");
			System.err.println("Můžete ho nainstalovat pomocí:");
			System.err.println("\t$ aptitude install graphviz   # (Debian/Ubuntu)");
			System.err.println("\t$ yum install graphviz        # (Fedora/RedHat)");
			return null;
		}
	}

	/**
	 * Převede text ve wiki syntaxi do XHTML.
	 * @param wiki vstupní text v dané wiki syntaxi
	 * @param syntaxe null nebo volitelně syntaxe (markdown, texy)
	 * @return naformátované XHTML
	 */
	public static String formátujWiki(String wiki, String syntaxe) throws IOException {
		if (isPříkazDostupný(PŘÍKAZ_MARKDOWN)) {
			Runtime r = Runtime.getRuntime();
			Process p = r.exec(new String[]{PŘÍKAZ_MARKDOWN});

			/**
			 * TODO: oříznout mezery na začátcích řádků, pokud je jich všude stejně?
			 * (odsazení v XML)
			 */
			PrintStream vstupProcesu = new PrintStream(p.getOutputStream());
			vstupProcesu.print(wiki);
			vstupProcesu.close();

			String chyby = načtiProud(p.getErrorStream());
			String xhtml = načtiProud(p.getInputStream());

			if (chyby.length() == 0) {
				return xhtml;
			} else {
				System.err.print("Při zpracování wiki syntaxe došlo k chybě: " + chyby);
				return null;
			}
		} else {
			System.err.println("Příkaz " + PŘÍKAZ_MARKDOWN + " není na vašem systému dostupný → nelze formátovat texty ve wiki syntaxi.");
			System.err.println("Můžete ho nainstalovat pomocí:");
			System.err.println("\t$ aptitude install markdown         # (Debian/Ubuntu)");
			System.err.println("\t$ yum install perl-Text-Markdown    # (Fedora/RedHat)");
			return null;
		}
	}
}
