/**
 * XML Web generátor – program na generování webových stránek
 * Copyright © 2012 František Kučera (frantovo.cz)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package cz.frantovo.xmlWebGenerator.makra;

import java.io.IOException;
import java.io.PrintStream;
import static cz.frantovo.xmlWebGenerator.NástrojeCLI.*;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * Wiki syntaxe
 *
 * @author František Kučera (frantovo.cz)
 */
public class Wiki {

	public enum SYNTAXE {

		markdown,
		texy
	}
	private static final String PŘÍKAZ_MARKDOWN = "markdown";
	/**
	 * Zde číhá tento PHP skript:
	 * https://hg.frantovo.cz/nekurak.net/file/tip/php/texy/http/index.php
	 */
	private static final String URL_TEXY = "http://nekurak.net/texy/http/";

	/**
	 * Převede text ve wiki syntaxi do XHTML.
	 *
	 * @param wiki vstupní text v dané wiki syntaxi
	 * @param syntaxe null nebo volitelně syntaxe (markdown, texy)
	 * @return naformátované XHTML
	 * TODO:
	 * - vracet místo textu instanci com.icl.saxon.om.NodeInfo
	 * http://saxon.sourceforge.net/saxon6.5.3/extensibility.html
	 * - nebo kontrolovat validitu vygenerovaného kódu (v současnosti se spoléháme na bezchybnost
	 * markdownu případně texy)
	 *
	 */
	public static String formátujWiki(String wiki, String syntaxe) throws IOException {
		if (syntaxe == null || SYNTAXE.valueOf(syntaxe) == SYNTAXE.markdown) {
			return formátujMarkdown(wiki);
		} else if (SYNTAXE.valueOf(syntaxe) == SYNTAXE.texy) {
			return formátujTexy(wiki);
		} else {
			throw new IllegalArgumentException("Syntaxe není podporovaná: " + syntaxe);
		}
	}

	private static String formátujMarkdown(String wiki) throws IOException {
		if (isPříkazDostupný(PŘÍKAZ_MARKDOWN)) {
			Runtime r = Runtime.getRuntime();
			Process p = r.exec(new String[]{PŘÍKAZ_MARKDOWN});

			/**
			 * TODO: oříznout mezery na začátcích řádků, pokud je jich všude stejně?
			 * (odsazení v XML)
			 */
			PrintStream vstupProcesu = new PrintStream(p.getOutputStream());
			vstupProcesu.print(wiki);
			vstupProcesu.close();

			String chyby = načtiProud(p.getErrorStream());
			String xhtml = načtiProud(p.getInputStream());

			if (chyby.length() == 0) {
				return xhtml;
			} else {
				System.err.print("Při zpracování wiki syntaxe došlo k chybě: " + chyby);
				return null;
			}
		} else {
			System.err.println("Příkaz " + PŘÍKAZ_MARKDOWN + " není na vašem systému dostupný → nelze formátovat texty ve wiki syntaxi.");
			System.err.println("Můžete ho nainstalovat pomocí:");
			System.err.println("\t$ aptitude install markdown         # (Debian/Ubuntu)");
			System.err.println("\t$ yum install perl-Text-Markdown    # (Fedora/RedHat)");
			return null;
		}
	}

	/**
	 * Texy! syntaxe je experimentální a oficiálně nepodporovaná.
	 *
	 * TODO: až bude balíček texy pro GNU/Linuxové distribuce:
	 * http://forum.texy.info/cs/873-balicek-pro-linuxove-distribuce
	 * řešit stejně jako Markdown.
	 */
	private static String formátujTexy(String wiki) throws IOException {
		System.out.println("Pozor: Texy! wiki syntaxe je experimentální a oficiálně nepodporovaná.");
		System.out.println("Pozor: používáte na vlastní nebezpečí!");
		System.out.println("Pozor: text k interpretování bude odeslán na vzdálené URL: " + URL_TEXY);
		System.out.println("Pokračovat? [a/N]");
		int pokračovat = System.in.read();

		if (pokračovat == 'a') {
			OutputStreamWriter požadavek = null;
			BufferedReader odpověď = null;			
			try {
				URL url = new URL(URL_TEXY);
				URLConnection spojeni = url.openConnection();
				spojeni.setDoOutput(true);

				/** Odešleme data */
				požadavek = new OutputStreamWriter(spojeni.getOutputStream());
				požadavek.write(URLEncoder.encode(wiki, StandardCharsets.UTF_8.name()));
				požadavek.flush();

				/** Přijmeme odpověď */
				odpověď = new BufferedReader(new InputStreamReader(spojeni.getInputStream(), StandardCharsets.UTF_8.name()));
				StringBuilder vysledek = new StringBuilder();
				String radka;
				while ((radka = odpověď.readLine()) != null) {
					vysledek.append(radka);
					vysledek.append("\n");
				}

				return vysledek.toString();
			} catch (Exception e) {
				throw new RuntimeException("Chyba při zpracovávání Texy! syntaxe: " + wiki, e);
			} finally {
				try {
					požadavek.close();
				} catch (IOException e) {
					e.printStackTrace(System.err);
				}
				try {
					odpověď.close();
				} catch (IOException e) {
					e.printStackTrace(System.err);
				}
			}
		} else {
			String hláška = "Texy! wiki syntaxe nebyla interpretována. Zdrojový text nebyl nikam odeslán.";
			System.out.println(hláška);
			return "<!-- " + hláška + " -->";
		}
	}
}
